<?php

if (class_exists('WP_Customize_Control')) {
    class Themexen_WP_Customize_Checkbox_Multiple extends WP_Customize_Control {
        public $type = 'checkbox-multiple';

        public function render_content() {
            if (empty($this->choices)) {
                return;
            }
            ?>
            <div class="mb-3">
                <label>
                    <span class="customize-control-title"><?php echo esc_html($this->label); ?></span>
                    <?php foreach ($this->choices as $value => $label) : ?>
                        <input type="checkbox" value="<?php echo esc_attr($value); ?>"
                            <?php checked( in_array( $value, explode( ',', $this->value() ) ) ); ?>
                               onchange="themexenCheckboxMultiSelect(this, '<?php echo esc_attr($this->id); ?>')">
                        <?php echo esc_html($label); ?>
                        <br>
                    <?php endforeach; ?>
                    <input type="hidden" id="<?php echo esc_attr($this->id); ?>" <?php $this->link(); ?> value="<?php echo esc_attr(implode(',', (array) $this->value())); ?>">
                </label>
                <span id="_customize-description-<?php echo esc_attr($this->id); ?>" class="description customize-control-description"><?php echo esc_html($this->description); ?></span>
                <script>
                    function themexenCheckboxMultiSelect(checkbox, id) {
                        let hiddenField = document.getElementById(id);
                        let values = hiddenField.value ? hiddenField.value.split(',') : [];
                        if (checkbox.checked) {
                            values.push(checkbox.value);
                        } else {
                            values = values.filter(v => v !== checkbox.value);
                        }
                        hiddenField.value = values.join(',');
                        hiddenField.dispatchEvent(new Event('change'));
                    }
                </script>
            </div>
            <?php
        }
    }
}


class Themxen_Persian {

    private $fonts = array(
        'iransans' => 'ایران سنس',
        'dana' => 'دانا',
        'vazirmatn' => 'وزیر',
        'yekanbakh' => 'یکان بخ',
        'parastoo' => 'پرستو',
        'yekan' => 'یکان'
    );

    private $version = '1.00.26';


    public function __construct(){
        add_action( 'customize_register', [$this, 'add_settings'] );
    }


    /**
     * @param $wp_customize
     * @return void
     *
     * Add and Register Setting to WP Customizer
     */
    public function add_settings( $wp_customize ){

        $wp_customize->add_section('themexen_fonts_section', array(
            'title'    => esc_html__( 'فونت فارسی | تمژن', 'woodmart' ),
            'priority' => 30,
        ));

        /**
         * To Enqueue Persian Fonts to Front of site
         */
        $wp_customize->add_setting('themexen_selected_fonts', array(
            'default'           => 'yekanbakh',
            'sanitize_callback' => [$this, 'sanitize_fonts_input'],
        ));

        $wp_customize->add_control(new Themexen_WP_Customize_Checkbox_Multiple(
            $wp_customize,
            'themexen_fonts_control',
            array(
                'label'    => __('فعالسازی فونت های فارسی برای قالب:', 'woodmart'),
                'description' => __('لطفا یک یا چند فونت را برای استفاده در قالب انتخاب کنید.', 'woodmart'),
                'section'  => 'themexen_fonts_section',
                'settings' => 'themexen_selected_fonts',
                'choices'  => $this->fonts,
            )
        ));


        /**
         * To Add Persian Font To WP Dashboard
         */
        $wp_customize->add_setting('themexen_admin_fonts_section', array(
            'default'           => array( 'none' ),
            'sanitize_callback' => [$this, 'sanitize_fonts_input'],
        ));

        $wp_customize->add_control(new WP_Customize_Control(
            $wp_customize,
            'themexen_fonts_wp_admin_control',
            array(
                'label'    => __('انتخاب فونت برای پیشخوان وردپرس:', 'woodmart'),
                'description' => __('شما می توانید برای پیشخوان وردپرس یک فونت انتخاب نمایید.', 'woodmart'),
                'section'  => 'themexen_fonts_section',
                'settings' => 'themexen_admin_fonts_section',
                'type'     => 'select', // تغییر نوع به select
                'choices'  => array(
                    'none'  => 'هیچ کدام',
                    'iransans' => 'ایران سنس',
                    'dana' => 'دانا',
                    'vazirmatn' => 'وزیر',
                    'yekanbakh' => 'یکان بخ',
                    'parastoo' => 'پرستو',
                    'yekan' => 'یکان'
                )
            )
        ));
    }


    /**
     * @param $input
     * @return string
     *
     * Sanitize Fonts Field
     *
     */
    public function sanitize_fonts_input( $input ) {

        $valid_fonts[] = "none";
        foreach ( $this->fonts as $key => $value ){
            $valid_fonts[] = $key;
        }

        if (is_string($input)) {
            $input = explode(',', $input);
        }

        $selected_fonts = array();
        if (is_array($input)) {
            foreach ($input as $font) {
                if (in_array($font, $valid_fonts)) {
                    $selected_fonts[] = $font;
                }
            }
        }

        return implode(',', $selected_fonts);
    }


    /**
     * @param $init_array
     * @return mixed
     *
     * Enqueue Fonts to Front of site
     */
    public function enqueue_fonts(){

        $fonts = get_theme_mod( 'themexen_selected_fonts', 'yekanbakh' );
        $fonts = explode( ',', $fonts );

        if ( is_array( $fonts ) && $fonts != array() ) {

            foreach ( $fonts as $font ){
                wp_register_style(
                    $font,
                    get_theme_file_uri( 'assets/persian-fonts/' .esc_attr( $font ). '/style.css' ),
                    array(),
                    $this->version
                );
                wp_enqueue_style( $font );
            }

        }

    }


    /**
     * @param $init_array
     * @return mixed
     *
     * Enqueue Fonts to Admin Dashboadr
     */
    public function enqueue_fonts_admin(){

        $font = get_theme_mod( 'themexen_admin_fonts_section' );
        if ( $font && $font[0] !== "none" ) {

            wp_register_style(
                $font,
                get_theme_file_uri( 'assets/persian-fonts/' .esc_attr( $font ). '/style-admin.css' ),
                array(),
                $this->version
            );
            wp_enqueue_style( $font );

        }

    }


    /**
     * @param $init_array
     * @return mixed
     *
     * Enqueu Fonts to Tiny MCE
     */
    public function enqueue_fonts_tinymce(){

        $font = get_theme_mod( 'themexen_admin_fonts_section' );
        if ( $font && $font[0] !== "none" ) {

            add_editor_style( 'assets/persian-fonts/' .esc_attr( $font ). '/style-admin.css' );

        }

    }


    /**
     * @param $init_array
     * @return mixed
     *
     * Add font to tiny mce
     */
    function add_to_tinymce($init_array) {

        $font = get_theme_mod( 'themexen_admin_fonts_section' );
        if ( $font && $font[0] !== "none" ) {

            $custom_fonts = $this->dictionary_fonts($font).', sans-serif;';

            // Add fonts to TinyMCE dropdown
            if (isset($init_array['font_formats'])) {
                $init_array['font_formats'] = $custom_fonts . $init_array['font_formats'];
            } else {
                $init_array['font_formats'] = $custom_fonts;
            }

        }

        return $init_array;
    }


    /**
     * Add Font Group to Elementor
     */
    function elementor_fonts_group( $font_groups ) {
        $fonts = get_theme_mod( 'themexen_selected_fonts', 'yekanbakh' );
        $fonts = explode( ',', $fonts );
        ;

        if ( is_array( $fonts ) && $fonts != array() ) {

            foreach ($fonts as $font) {
                $font_groups[$font] = __( $font );

            }
        }

        return $font_groups;
    }

    /**
     * @param $fonts
     * @return mixed
     *
     * Add Font to elementor area
     *
     */
    function add_fonts_to_elementor( $additional_fonts ) {

        $fonts = get_theme_mod( 'themexen_selected_fonts', 'yekanbakh' );
        $fonts = explode( ',', $fonts );

        if ( is_array( $fonts ) && $fonts != array() ) {

            foreach ($fonts as $font) {
                $additional_fonts[$this->dictionary_fonts( $font )] = $font;
            }
        }

        return $additional_fonts;
    }


    /**
     * @param $fonts
     * @return mixed
     *
     * Add Font to JS-Composer (WP Bakary)
     *
     */
    function add_fonts_to_jscomposer( $fonts_list ) {

        $fonts = get_theme_mod( 'themexen_selected_fonts', 'yekanbakh' );
        $fonts = explode( ',', $fonts );

        if ( is_array( $fonts ) && $fonts != array() ) {

            foreach ( $fonts as $font ){

                $fonts_list[$this->dictionary_fonts( $font )] = $this->dictionary_fonts( $font ).', sans-serif';

                return $fonts_list;
            }

        }

        return $fonts_list;
    }


    /**
     * @param $font
     * @return string
     *
     * Convert name of font to @font-face
     *
     */
    private function dictionary_fonts( $font ){

        switch ( $font ) {
            case 'iransans':
                $style = 'IRANSansX';
                break;
            case 'dana':
                $style = 'Dana';
                break;
            case 'parastoo':
                $style = 'Parastoo';
                break;
            case 'vazirmatn':
                $style = 'Vazirmatn';
                break;
            case 'yekan':
                $style = 'Yekan';
                break;
            case 'yekanbakh':
                $style = 'YekanBakh';
                break;
            default:
                $style = 'YekanBakh';
        }

        return $style;


    }


    /**
     * @param $init_array
     * @return mixed
     *
     * Enqueue style to Front of site
     */
    public function enqueue_style(){

        wp_register_style(
            'themexen-woodmart-rtl',
            get_theme_file_uri( 'assets/persian-style/style-rtl.css' ),
            array( 'woodmart-style' ),
            $this->version
        );
        wp_enqueue_style( 'themexen-woodmart-rtl' );

    }


    /**
     * @param $init_array
     * @return mixed
     *
     * Enqueue Fonts to Admin Dashboadr
     */
    public function enqueue_style_admin(){

        wp_register_style(
            'themexen-admin-woodmart-rtl',
            get_theme_file_uri( 'assets/persian-style/admin.css' ),
            array(),
            $this->version
        );
        wp_enqueue_style( 'themexen-admin-woodmart-rtl' );

    }


    /**
     * @param $font
     * @return array
     *
     * Custom callback for adding Fonts to woodmart theme settings
     *
     */
    public function setting_custom_fonts ( $opt, $slug ){

        if ( $slug == "multi_custom_fonts" ){

            $fonts = get_theme_mod( 'themexen_selected_fonts', 'yekanbakh' );
            $fonts = explode( ',', $fonts );

            if ( !is_array($opt) ){
                $opt = array();
            }


            if ( is_array( $fonts ) && $fonts != array() ) {

                foreach ( $fonts as $font ){
                    $opt[] = array(
                        "font-name" => $this->dictionary_fonts( $font ),
                    );
                }

            }

        }

        return $opt;
    }




    /**
     * @return void
     *
     * Run and Ready for Persian language
     *
     */
    public function run(){

        // To Support Persian Fonts
        add_action( 'wp_enqueue_scripts', [$this, 'enqueue_fonts'] );
        add_action( 'admin_enqueue_scripts', [$this, 'enqueue_fonts_admin'] );
        add_action( 'admin_init', [$this, 'enqueue_fonts_tinymce'] );
        add_filter( 'tiny_mce_before_init', [$this, 'add_to_tinymce'] );
        if ( class_exists( '\Elementor\Plugin' ) ) {
            add_filter( 'elementor/fonts/groups', [$this, 'elementor_fonts_group'], 10, 1 );
            add_filter( 'elementor/fonts/additional_fonts', [$this, 'add_fonts_to_elementor'], 10, 1 );
        }
        if ( class_exists( 'Vc_Manager' ) ) {
            add_filter( 'vc_google_fonts_get_fonts_filter', [$this, 'add_fonts_to_jscomposer'] );
        }

        // To Support Right To Left
        add_action( 'admin_enqueue_scripts', [$this, 'enqueue_style_admin'] );
        add_action( 'wp_enqueue_scripts', [$this, 'enqueue_style'] );

        // Just for woodmart theme:
        add_filter( 'woodmart_option', [$this, 'setting_custom_fonts'], 99, 2 );



    }

}

$test = new Themxen_Persian();
$test->run();
